using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using System.Xml;
using System.Collections;
using System.Collections.Specialized;
using System.Diagnostics;
using Microsoft.EnterpriseSingleSignOn.Interop;

namespace BizTalk.Tools.SSOStorage
{
    public partial class ApplicationManager : Form
    {
        public ApplicationManager()
        {
            InitializeComponent();
        }

        private void btnCreateApp_Click(object sender, EventArgs e)
        {

            //write to SSOPropBag to figure out how many rows we have
            SSOPropBag propertiesBag = new SSOPropBag();
            ArrayList maskArray = new ArrayList();

            foreach (DataGridViewRow row in dgvFields.Rows)
            {
                if (row.Cells[0].Value != null)
                {
                    //set values
                    object objPropValue = "<empty>";
                    propertiesBag.Write(row.Cells[0].Value.ToString(), ref objPropValue);

                    //store mask
                    if (row.Cells[1].FormattedValue.ToString() == "yes")
                    {
                        maskArray.Add(SSOFlag.SSO_FLAG_FIELD_INFO_MASK);
                    }
                    else
                    {
                        maskArray.Add(0);
                    }  
                }
            }

            try
            {
                //create and enable application
                SSOConfigManager.CreateConfigStoreApplication(txtAppName.Text, txtDescription.Text, txtAppUserAcct.Text, txtAppAdminAcct.Text, propertiesBag, maskArray);

                //set default configuration field values
                SSOConfigManager.SetConfigProperties(txtAppName.Text, propertiesBag);

                //show success
                MessageBox.Show("Application Successfully Created", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information);

                //ResetNewTab(false);
            }
            catch (Exception ex)
            {
                MessageBox.Show("Error Occured.  Details: " + ex.ToString(), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void btnRetrieveSettings_Click(object sender, EventArgs e)
        {
            dgvFieldsEdit.Rows.Clear();
            
            string appName = txtAppQueryName.Text;

            if (appName != null || appName != string.Empty)
            {
                try
                {
                    string appUserAcct, appAdminAcct, description, contactInfo;

                    HybridDictionary props = SSOConfigManager.GetConfigProperties(appName, out description, out contactInfo, out appUserAcct, out appAdminAcct);
                    txtDescriptionQuery.Text = description;
                    txtAppAdminAcctQuery.Text = appAdminAcct;
                    txtAppUserAccountQuery.Text = appUserAcct;

                    foreach (DictionaryEntry appProp in props)
                    {
                        int index = dgvFieldsEdit.Rows.Add();
                        DataGridViewRow dgvr = dgvFieldsEdit.Rows[index];

                        dgvr.Cells[0].Value = appProp.Key.ToString();
                        dgvr.Cells[1].Value = appProp.Value.ToString();

                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show("Error Occured.  Details: " + ex.ToString(), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            else
            {
                MessageBox.Show("Please enter an application name.", "Warning", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        private void btnDeleteApp_Click(object sender, EventArgs e)
        {
            if (txtAppQueryName.Text != null || txtAppQueryName.Text != string.Empty)
            {

                if (MessageBox.Show("Are you sure you wish to delete this application?", "Warning", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Warning) == DialogResult.Yes)
                {
                    try
                    {
                        SSOConfigManager.DeleteApplication(txtAppQueryName.Text);

                        //confirmation
                        MessageBox.Show("Application deleted.", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information);

                        ResetManageTab(true);
                    }
                    catch (Exception ex)
                    {
                        MessageBox.Show("Error Occured.  Details: " + ex.ToString(), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
            else
            {
                MessageBox.Show("Please enter an application name for deletion.", "Warning", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        private void btnSave_Click(object sender, EventArgs e)
        {
            SSOPropBag propertiesBag = new SSOPropBag();

            foreach (DataGridViewRow row in dgvFieldsEdit.Rows)
            {
                if (row.Cells[0].Value != null)
                {
                    //set values
                    string propName = row.Cells[0].Value.ToString();
                    object objPropValue = row.Cells[1].Value;
                
                    propertiesBag.Write(propName, ref objPropValue);
                }
            }

            try
            {
                SSOConfigManager.SetConfigProperties(txtAppQueryName.Text, propertiesBag);

                MessageBox.Show("Properties saved successfully.", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information);
            }
            catch (Exception ex)
            {
                MessageBox.Show("Error Occured.  Details: " + ex.ToString(), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void ResetNewTab(bool resetAppName)
        {
            if (resetAppName)
            {
                txtAppName.Text = "";
            }
            txtDescription.Text = "";
            txtAppUserAcct.Text = "";
            txtAppAdminAcct.Text = "";
            dgvFields.Rows.Clear();
        }

        private void ResetManageTab(bool resetAppName)
        {
            if (resetAppName)
            {
                txtAppQueryName.Text = "";
            }
                txtDescriptionQuery.Text = "";
                txtAppUserAccountQuery.Text = "";
                txtAppAdminAcctQuery.Text = "";
            
            dgvFieldsEdit.Rows.Clear();
        }

        private void saveConfigurationToXMLToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (txtAppName.Text == null || txtAppName.Text == string.Empty)
            {
                MessageBox.Show("Please enter an application name", "Warning", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
            else
            {
                if (fbdSavePath.ShowDialog() == DialogResult.OK)
                {
                    string path = fbdSavePath.SelectedPath;

                    SaveXmlConfiguration(path);

                    MessageBox.Show("Configuration saved successfully.", "Information", MessageBoxButtons.OK, MessageBoxIcon.Information);

                }
            }
        }

        private void SaveXmlConfiguration(string path)
        {
            //generate temp guid
            string tempFileName = path + "\\SSO_" + txtAppName.Text + "_" + System.Guid.NewGuid().ToString() + ".xml";

            XmlTextWriter writer = new XmlTextWriter(tempFileName, Encoding.UTF8);

            //start document
            writer.WriteStartDocument();

            //start root
            writer.WriteStartElement("sso");

            writer.WriteStartElement("application");
            writer.WriteAttributeString("name", txtAppName.Text);


            writer.WriteElementString("description", txtDescription.Text);
            writer.WriteElementString("appUserAccount", txtAppUserAcct.Text);
            writer.WriteElementString("appAdminAccount", txtAppAdminAcct.Text);

            int ordinalCount = 0;
            ArrayList fieldList = new ArrayList();
            //loop through field list
            foreach (DataGridViewRow row in dgvFields.Rows)
            {
                if (row.Cells[0].Value != null)
                {
                    writer.WriteStartElement("field");
                    writer.WriteAttributeString("ordinal", ordinalCount.ToString());
                    writer.WriteAttributeString("label", row.Cells[0].Value.ToString());
                    writer.WriteAttributeString("masked", row.Cells[1].FormattedValue.ToString());
                    writer.WriteEndElement();

                    fieldList.Add(row.Cells[0].Value.ToString());
                    ordinalCount++;
                }
            }

            writer.WriteStartElement("flags");
            writer.WriteAttributeString("configStoreApp", "yes");
            writer.WriteAttributeString("allowLocalAccounts", "yes");
            writer.WriteAttributeString("enableApp", "yes");
            writer.WriteEndElement();

            //close application
            writer.WriteEndElement();

            //close root
            writer.WriteEndElement();

            writer.Flush();
            writer.Close();
        }

        private void loadConfigurationFromXMLToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (ofdConfigFile.ShowDialog() == DialogResult.OK)
            {
                string path = ofdConfigFile.FileName;

                LoadXmlConfigFile(path);
            }
        }

        private void LoadXmlConfigFile(string path)
        {
            XmlDocument configDoc = new XmlDocument();
            configDoc.Load(path);

            string appName = configDoc.SelectSingleNode("//application/@name").InnerText;
            txtAppName.Text = appName;

            string description = configDoc.SelectSingleNode("//description").InnerText;
            txtDescription.Text = description;

            string appUserAcct = configDoc.SelectSingleNode("//appUserAccount").InnerText;
            txtAppUserAcct.Text = appUserAcct;

            string appAdminAcct = configDoc.SelectSingleNode("//appAdminAccount").InnerText;
            txtAppAdminAcct.Text = appAdminAcct;

            //grab fields
            XmlNodeList fields = configDoc.SelectNodes("//field");

            foreach(XmlNode field in fields)
            {
                int index = dgvFields.Rows.Add();
                DataGridViewRow dgvr = dgvFields.Rows[index];

                dgvr.Cells[0].Value = field.Attributes["label"].InnerText;
                dgvr.Cells[1].Value = field.Attributes["masked"].InnerText;
            }


        }

        private void clearThisTabToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (tcManagement.SelectedTab.Name == "tpCreate")
            {
                ResetNewTab(true);
            }
            else
            {
                ResetManageTab(true);
            }
        }

        private void tcManagement_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (tcManagement.SelectedTab.Name == "tpManage")
            {
                saveConfigurationToXMLToolStripMenuItem.Enabled = false;
                loadConfigurationFromXMLToolStripMenuItem.Enabled = false;
            }
            else
            {
                saveConfigurationToXMLToolStripMenuItem.Enabled = true;
                loadConfigurationFromXMLToolStripMenuItem.Enabled = true;
            }
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void aboutMeToolStripMenuItem_Click(object sender, EventArgs e)
        {
            About aboutForm = new About();

            aboutForm.ShowDialog();
        }

    }
}